import math
import numpy as np


def thomas(first, last, a, b, c, d):
    """トーマス法を用いて3項方程式
       ( a(i)x(i-1) + b(i)x(i) + c(i)x(i+1) = d(i) )を解く．
       計算進行に伴って配列bにb，g，配列dにd，s，xを保持することで，
       係数aと係数cが同じ場合のthomas()で使用するスペースを節約．

        引数:
            first (array): 方程式をfirstからlastについて解く．
            last (array): 方程式をfirstからlastについて解く．
            a (array): 差分方程式のU[i-1](i=2~N)の係数を記憶する配列．
            b (array): 初期にb, 計算進行に伴ってgを記憶する配列．
            c (array): 差分方程式のU[i+1](i=2~N)の係数を記憶する配列．
            d (array): 初期にd, 計算進行に伴ってsさらにxを記憶する配列．
    """
    start = first + 1
    for i in range(start, last):  # 3項方程式をトーマス法を用いて解く
        p = c[i] / b[i - 1]  # 右辺のbは本文ではg
        b[i] = b[i] - p * a[i - 1]  # 左辺のbは本文ではg
        d[i] = d[i] - p * d[i - 1]  # 左辺と右辺第2項のdは本文ではs
    d[last - 1] = d[last - 1] / b[last - 1]
    for i in range(last - start, 0, -1):
        d[i] = (d[i] - a[i] * d[i + 1]) / b[i]  # 左辺と右辺第2項のdは本文ではx,
                                                # 右辺のbは本文ではg


mesh_max = int(input("格子点数を入力してください. (e.g. 20): "))
delta_x = 1 / mesh_max  # 格子幅
A = np.zeros(mesh_max)  # 差分方程式のU[i-1] (i=2~N)の係数を記憶する配列
B = np.zeros(mesh_max)  # 差分方程式のU[i] (i=2~N)の係数を記憶する配列
C = np.zeros(mesh_max)  # 差分方程式のU[i+1] (i=2~N)の係数を記憶する配列
D = np.zeros(mesh_max)  # 差分方程式の右辺の値を記憶する配列，
                        # thomas()を呼んだあとでは近似解が記憶される

for i in range(1, mesh_max):  # 常微分方程式の境界値問題を解く
    A[i] = 1
    B[i] = delta_x ** 2 - 2
    C[i] = 1
    D[i] = -i * delta_x ** 3

thomas(1, mesh_max, A, B, C, D)
print(' ' * 7 + "x", ' ' * 7 + "approx", ' ' * 5 + "exact", ' ' * 5 + "error")
for i in range(1, mesh_max):
    x = delta_x * i  # 格子点のＸ座標
    exact = math.sin(x) / math.sin(1.0) - x  # 厳密解
    err_diff = (D[i] - exact) / exact * 100  # 厳密解との誤差(％)
    print("{:11.6f}".format(x),
          "{:11.6f}".format(D[i]), "{:11.6f}".format(exact),\
          "{:11.6f}".format(err_diff))