import matplotlib.animation as animation
import matplotlib.pyplot as plt
import numpy as np


def thomas(first, last, a, b, c, d):
    """トーマス法を用いて3項方程式
    ( a(i)x(i-1) + b(i)x(i) + c(i)x(i+1) = d(i) ) を解く．
    計算進行に伴って配列bにb，g，
    配列dにd，s，xを保持することで，
    係数aと係数cが同じ場合のthomas()で使用するスペースを節約．

        引数:
            first (array): 方程式をfirstからlastについて解く．
            last (array): 方程式をfirstからlastについて解く．
            a (array): 差分方程式のU[i-1]（i=2~N）の係数を記憶する配列．
            b (array): 初期にb, 計算進行に伴ってgを記憶する配列．
            c (array): 差分方程式のU[i+1]（i=2~N）の係数を記憶する配列．
            d (array): 初期にd, 計算進行に伴ってsさらにxを記憶する配列．
    """
    start = first + 1
    for i in range(start, last):  # 3項方程式をトーマス法を用いて解く
        p = c[i] / b[i - 1]  # 右辺のbは本文ではg
        b[i] = b[i] - p * a[i - 1]  # 左辺のbは本文ではg
        d[i] = d[i] - p * d[i - 1]  # 左辺と右辺第2項のdは本文ではs
    d[last - 1] = d[last - 1] / b[last - 1]
    for i in range(last - start, 0, -1):
        d[i] = (d[i] - a[i] * d[i + 1]) / b[i]  # 左辺と右辺第2項のdは本文ではx,
                                                # 右辺のbは本文ではg


mesh_max = int(input("格子点数を入力してください (<51). (e.g. 40): ")) + 1
time_max = int(input("計算打ち切りのタイムステップ数を入力してください."\
    "(e.g. 50): "))
delta_t = float(input("時間間隔Δtを入力してください. (e.g. 0.01): "))
delta_x = 1 / (mesh_max - 1)
r = delta_t / (delta_x ** 2)
mid_mesh_point = mesh_max // 2
fig = plt.figure()
graphs = []  # アニメーション化する際に必要となるグラフを保存する配列
A = np.zeros(mesh_max)  # 差分方程式のU[i-1] （i=2~N）の係数を記憶する配列
B = np.zeros(mesh_max)  # 差分方程式のU[i] （i=2~N）の係数を記憶する配列
C = np.zeros(mesh_max)  # 差分方程式のU[i+1] （i=2~N）の係数を記憶する配列
D = np.zeros(mesh_max)  # 差分方程式の右辺の値を記憶する配列，
                        # thomas()を呼んだあとでは近似解が記憶される
U = np.zeros(mesh_max)  # 現時点での解を記憶する配列

for i in range(0, mesh_max):  # 初期化
    X = i / (mesh_max - 1)
    if i <= mid_mesh_point:
        U[i] = X
    else:
        U[i] = 1.0 - X

for t in range(0, time_max):  # 1次元熱伝導方程式をオイラー陰解法を用いて解く
    if t % 10 == 0:  # 10ステップに1回，アニメーション化する際に
                     # 必要となるグラフを記録
        graph = plt.plot(U, color="red")
        graphs.append(graph)

    U[0] = 0.0  # 境界条件（左側）
    U[mesh_max - 1] = 0.0  # 境界条件（右側）

    for i in range(1, mesh_max - 1):  # アルゴリズムに従って3項方程式の係数を計算
        A[i] = r
        B[i] = -2 * r - 1
        C[i] = r
        D[i] = -U[i]

    thomas(1, mesh_max - 1, A, B, C, D)

    for i in range(1, mesh_max - 1):
        U[i] = D[i]

graph = plt.plot(U, color="red")
graphs.append(graph)
animated_result = animation.ArtistAnimation(fig, graphs)
plt.show()
