import math
import matplotlib.cm as cm
import matplotlib.pyplot as plt
import numpy as np


def grid(jmax, kmax, x, y):
    """格子を生成する．

        引数:
            jmax, kmax (int): ξ, η方向の格子点数．
            x, y (array): 格子点のＸ, Y座標を記憶する配列．
    """

    aa = 0.75
    bb = 1
    hh = 0.025
    ra = 1.15
    rr = np.zeros(kmax)
    rr[0] = 1

    for k in range(1, kmax):
        rr[k] = rr[k - 1] + hh * ra ** k
    for k in range(0, kmax):
        for j in range(0, jmax):
            tt = 2 * math.pi * (j - 1) / (jmax - 2)
            bcc = bb + (aa - bb) * k / (kmax - 1)
            x[j, k] = aa * rr[k] * math.cos(tt)
            y[j, k] = bcc * rr[k] * math.sin(tt)


def metj(jmax, kmax, jm, km, ityp, x, y, xx, xy, yx, yy, c1, c2, c3, c4, c5, aj):
    """変換の係数を計算する．

        引数:
            jmax, kmax (int): ξ, η方向の格子点数．
            jm, km (int): jmax-1, kmax-1．
            X, Y (array): 格子点のＸ, Ｙ座標を記憶する配列．
            ityp (int): 1->極座標のように1方向に周期的な座標を用いる場合, 
                        0->普通の座標系を用いる場合．
            x, y (array): 格子点のＸ, Ｙ座標を記憶する配列．
            xx, xy, yx, yy (array): それぞれξX, ξY, ηX, ηYを記憶する配列．
            c1, c2, c3, c4, c5 (array): Δ〇=C1*〇ξξ+C2*〇ξη+C3*〇ηη+
                C4*〇ξ+C5*〇ηと表したときの各係数を記憶する配列．
            aj (array): 変換のヤコビアンJ．
    """
    for k in range(0, kmax):
        for j in range(0, jmax):
            if k == 0:
                xe = 0.5 * (-x[j, 2] + 4 * x[j, 1] - 3 * x[j, 0])
                ye = 0.5 * (-y[j, 2] + 4 * y[j, 1] - 3 * y[j, 0])
            elif k == kmax - 1:
                xe = 0.5 * (x[j, kmax - 3] -
                        4 * x[j, kmax - 2] + 3 * x[j, kmax - 1])
                ye = 0.5 * (y[j, kmax - 3] -
                        4 * y[j, kmax - 2] + 3 * y[j, kmax - 1])
            else:
                xe = 0.5 * (x[j, k + 1] - x[j, k - 1])
                ye = 0.5 * (y[j, k + 1] - y[j, k - 1])

            if j == 0:
                xxi = 0.5 * (-x[2, k] + 4 * x[1, k] - 3 * x[0, k])
                yxi = 0.5 * (-y[2, k] + 4 * y[1, k] - 3 * y[0, k])
                if ityp == 1:
                    xxi = 0.5 * (x[1, k] - x[jmax - 3, k])
                    yxi = 0.5 * (y[1, k] - y[jmax - 3, k])
            elif j == jmax - 1:
                xxi = 0.5 * (x[jmax - 3, k] -
                         4 * x[jmax - 2, k] + 3 * x[jmax - 1, k])
                yxi = 0.5 * (y[jmax - 3, k] -
                         4 * y[jmax - 2, k] + 3 * y[jmax - 1, k])
                if ityp == 1:
                    xxi = 0.5 * (x[2, k] - x[jmax - 2, k])
                    yxi = 0.5 * (y[2, k] - y[jmax - 2, k])
            else:
                xxi = 0.5 * (x[j + 1, k] - x[j - 1, k])
                yxi = 0.5 * (y[j + 1, k] - y[j - 1, k])

            if ityp == 1 and j == 0:
                xxi = 0.5 * (x[j + 1, k] - x[jm - 2, k])
                yxi = 0.5 * (y[j + 1, k] - y[jm - 2, k])

            if ityp == 1 and j == jmax - 1:
                xxi = 0.5 * (x[2, k] - x[j - 1, k])
                yxi = 0.5 * (y[2, k] - y[j - 1, k])

            ajj = xxi * ye - xe * yxi  # 変換のヤコビアン
            xx[j, k] = ye / ajj
            yx[j, k] = -yxi / ajj
            xy[j, k] = -xe / ajj
            yy[j, k] = xxi / ajj
            aj[j, k] = ajj

    for k in range(0, kmax):
        for j in range(0, jmax):
            c1[j, k] = xx[j, k] ** 2 + xy[j, k] ** 2
            c3[j, k] = yx[j, k] ** 2 + yy[j, k] ** 2
            c2[j, k] = 2 * (xx[j, k] * yx[j, k] + xy[j, k] * yy[j, k])

    for k in range(1, km):
        for j in range(1, jm):
            c77 = xx[j, k] * (xx[j + 1, k] - xx[j - 1, k]) + \
                  yx[j, k] * (xx[j, k + 1] - xx[j, k - 1]) + \
                  xy[j, k] * (xy[j + 1, k] - xy[j - 1, k]) + \
                  yy[j, k] * (xy[j, k + 1] - xy[j, k - 1])
            c88 = xx[j, k] * (yx[j + 1, k] - yx[j - 1, k]) + \
                  yx[j, k] * (yx[j, k + 1] - yx[j, k - 1]) + \
                  xy[j, k] * (yy[j + 1, k] - yy[j - 1, k]) + \
                  yy[j, k] * (yy[j, k + 1] - yy[j, k - 1])
            c4[j, k] = c77 * 0.5
            c5[j, k] = c88 * 0.5


def init(jmax, kmax, u, v, p, alp):
    """初期条件（迎角 ALPH の一様流）を与える．

        引数:
            jmax, kmax (int): ξ, η方向の格子点数．
            u, v (array): Ｘ, Y方向の速度を記憶する配列．
            p (array): 圧力を記憶する配列．
            alp (int): 迎角（度）．
    """

    ttt = alp * math.pi / 180  # 迎角（ラジアン）を計算
    for k in range(0, kmax):
        for j in range(0, jmax):
            u[j, k] = math.cos(ttt)
            v[j, k] = math.sin(ttt)
            p[j, k] = -0.5 * (u[j, k] ** 2 + v[j, k] ** 2)


def rhs(jm, km, xx, xy, yx, yy, u, v, q, dti):
    """ポアソン方程式の右辺を計算する．

        引数:
            jm, km (int): jmax-1, kmax-1．
            xx, xy, yx, yy (array): それぞれξX, ξY, ηX, ηYを記憶する配列．
            u, v (array): Ｘ, Y方向の速度を記憶する配列．
            q (array): ポアソン方程式の右辺を記憶する配列．
            dti (float): 1/時間間隔．
    """
    for k in range(1, km):
        for j in range(1, jm):
            uxd = xx[j, k] * (u[j + 1, k] - u[j - 1, k]) + \
                  yx[j, k] * (u[j, k + 1] - u[j, k - 1])
            uyd = xy[j, k] * (u[j + 1, k] - u[j - 1, k]) + \
                  yy[j, k] * (u[j, k + 1] - u[j, k - 1])
            vxd = xx[j, k] * (v[j + 1, k] - v[j - 1, k]) + \
                  yx[j, k] * (v[j, k + 1] - v[j, k - 1])
            vyd = xy[j, k] * (v[j + 1, k] - v[j - 1, k]) + \
                  yy[j, k] * (v[j, k + 1] - v[j, k - 1])
            q[j, k] = -0.25 * (uxd * uxd + 2 * uyd * vxd + vyd * vyd) + \
                      0.5 * (uxd + vyd) * dti


def prsbc(jmax, kmax, jm, ityp, xx, xy, c2, c3, c5, aj, u, v, p, rei):
    """圧力の境界条件を与える．

        引数:
            jmax, kmax (int): ξ, η方向の格子点数．
            jm (int): jmax-1．
            ityp (int): 1->極座標のように1方向に周期的な座標を用いる場合,
                        0->普通の座標系を用いる場合．
            xx, xy (array): それぞれξX, ξYを記憶する配列．
            c1, c2, c3, c4, c5 (array): Δ〇=C1*〇ξξ+C2*〇ξη+C3*〇ηη+
                C4*〇ξ+C5*〇ηと表したときの各係数を記憶する配列．
            aj (array): 変換のヤコビアンJ．
            u, v (array): Ｘ, Y方向の速度を記憶する配列．
            p (array): 圧力を記憶する配列．
            rei (float): 1/レイノルズ数．
    """
    # no-slip at k = 1, free stream at k=kmax
    for j in range(1, jm):
        ul = 0.5 * c2[j, 0] * (u[j + 1, 1] - u[j - 1, 1]) + c5[j, 0] * u[j, 0]
        vl = 2 * c3[j, 0] * v[j, 1]
        p[j, 0] = p[j, 1] - rei * aj[j, 0] * (xx[j, 0] * vl - xy[j, 1] * ul)
        p[j, kmax - 1] = -0.5 * (u[j, kmax - 1] ** 2 + v[j, kmax - 1] ** 2)

    if ityp == 1:  # periodic
        for k in range(0, kmax):
            p[0, k] = p[jm - 1, k]
            p[jmax - 1, k] = p[1, k]
    else:  # extapolate
        for k in range(0, kmax):
            p[0, k] = p[1, k]
            p[jmax - 1, k] = p[jm - 1, k]


def prs(jm, km, c1, c2, c3, c4, c5, p, q, const):
    """圧力を計算する．

        引数:
            jm, km (int): jmax-1, kmax-1．
            c1, c2, c3, c4, c5 (array): Δ〇=C1*〇ξξ+C2*〇ξη+C3*〇ηη+
                C4*〇ξ+C5*〇ηと表したときの各係数を記憶する配列．
            p (array): 圧力を記憶する配列．
            q (array): ポアソン方程式の右辺を記憶する配列．
            const (float): 圧力に関するポアソン方程式の近似差分方程式を
                SOR法で解くときの加速係数．
    """
    err = 0
    for k in range(1, km):
        for j in range(1, jm):
            cc = 0.5 / (c1[j, k] + c3[j, k])
            pa = c1[j, k] * (p[j + 1, k] + p[j - 1, k]) + \
                 c3[j, k] * (p[j, k + 1] + p[j, k - 1]) + \
                 0.25 * c2[j, k] * (p[j + 1, k + 1] - p[j - 1, k + 1] -
                 p[j + 1, k - 1] + p[j - 1, k - 1]) + \
                 0.5 * c4[j, k] * (p[j + 1, k] - p[j - 1, k]) + \
                 0.5 * c5[j, k] * (p[j, k + 1] - p[j, k - 1])
            pp = (pa - q[j, k]) * cc
            err = err + (pp - p[j, k]) ** 2
            p[j, k] = p[j, k] * (1 - const) + pp * const
    return err


def ns(jm, km, xx, xy, yx, yy, c1, c2, c3, c4, c5, u, v, p, q, d, rei, dt):
    """速度を計算する．

        引数:
            jm, km (int): jmax-1, kmax-1．
            xx, xy, yx, yy (array): それぞれξX, ξY, ηX, ηYを記憶する配列．
            c1, c2, c3, c4, c5 (array): Δ〇=C1*〇ξξ+C2*〇ξη+C3*〇ηη+
                C4*〇ξ+C5*〇ηと表したときの各係数を記憶する配列．
            u, v (array): Ｘ, Y方向の速度を記憶する配列．
            p (array): 圧力を記憶する配列．
            d (array): 1つ後の時間ステップでのＸ方向の速度を記憶する配列．
            q (array): 1つ後の時間ステップでのY方向の速度を記憶する配列．
            rei (int): 1/レイノルズ数．
            dt (float): 時間間隔．
    """

    for k in range(1, km):
        for j in range(1, jm):
            # conservative form
            # unl = xx[j, k] * (u[j + 1, k] ** 2 - u[j - 1, k] ** 2) * 0.5 + \
            #       yx[j, k] * (u[j, k + 1] ** 2 - u[j, k - 1] ** 2) * 0.5 + \
            #       xy[j, k] * (u[j + 1, k] * v[j + 1, k] -
            #                   u[j - 1, k] * v[j - 1, k]) * 0.5 + \
            #       yy[j, k] * (u[j, k + 1] * v[j, k + 1] -
            #                   u[j, k - 1] * v[j, k - 1]) * 0.5
            # vnl = xy[j, k] * (v[j + 1, k] ** 2 - v[j - 1, k] ** 2) * 0.5 + \
            #       yy[j, k] * (v[j, k + 1] ** 2 - v[j, k - 1] ** 2) * 0.5 + \
            #       xx[j, k] * (u[j + 1, k] * v[j + 1, k] -
            #                   u[j - 1, k] * v[j - 1, k]) * 0.5 + \
            #       yx[j, k] * (u[j, k + 1] * v[j, k + 1] -
            #                   u[j, k - 1] * v[j, k - 1]) * 0.5
            # non-conservative form
            ua = (u[j, k] * xx[j, k] + v[j, k] * xy[j, k]) * 0.5
            va = (u[j, k] * yx[j, k] + v[j, k] * yy[j, k]) * 0.5
            unl = ua * (u[j + 1, k] - u[j - 1, k]) + \
                  va * (u[j, k + 1] - u[j, k - 1])
            vnl = ua * (v[j + 1, k] - v[j - 1, k]) + \
                  va * (v[j, k + 1] - v[j, k - 1])
            uvs = c1[j, k] * (u[j + 1, k] - 2 * u[j, k] + u[j - 1, k]) + \
                  c2[j, k] * (u[j + 1, k + 1] - u[j + 1, k - 1] -
                              u[j - 1, k + 1] + u[j - 1, k - 1]) * 0.25 + \
                  c3[j, k] * (u[j, k + 1] - 2 * u[j, k] + u[j, k - 1]) + \
                  0.5 * c4[j, k] * (u[j + 1, k] - u[j - 1, k]) + \
                  0.5 * c5[j, k] * (u[j, k + 1] - u[j, k - 1])
            vvs = c1[j, k] * (v[j + 1, k] - 2 * v[j, k] + v[j - 1, k]) + \
                  c2[j, k] * (v[j + 1, k + 1] - v[j + 1, k - 1] -
                              v[j - 1, k + 1] + v[j - 1, k - 1]) * 0.25 + \
                  c3[j, k] * (v[j, k + 1] - 2 * v[j, k] + v[j, k - 1]) + \
                  0.5 * c4[j, k] * (v[j + 1, k] - v[j - 1, k]) + \
                  0.5 * c5[j, k] * (v[j, k + 1] - v[j, k - 1])
            d[j, k] = u[j, k] + dt * (-unl - 
                      0.5 * xx[j, k] * (p[j + 1, k] - p[j - 1, k]) -
                      0.5 * yx[j, k] * (p[j, k + 1] - p[j, k - 1]) +
                      rei * uvs)
            q[j, k] = v[j, k] + dt * (-vnl - 
                      0.5 * xy[j, k] * (p[j + 1, k] - p[j - 1, k]) -
                      0.5 * yy[j, k] * (p[j, k + 1] - p[j, k - 1]) +
                      rei * vvs)

    for k in range(1, km):
        for j in range(1, jm):
            u[j, k] = d[j, k]
            v[j, k] = q[j, k]


def bc(jmax, kmax, jm, ityp, u, v, alp):
    """速度の境界条件を与える．

        引数:
            jmax, kmax (int): ξ, η方向の格子点数．
            jm (int): jmax-1．
            ityp (int): 1->極座標のように1方向に周期的な座標を用いる場合,
                        0->普通の座標系を用いる場合．
            u, v (array): Ｘ, Ｙ方向の速度を記憶する配列．
            alp (int): 迎角（度）．
    """
    ttt = alp * math.pi / 180
    for j in range(0, jmax):
        u[j, 0] = 0
        v[j, 0] = 0
        u[j, kmax - 1] = math.cos(ttt)
        v[j, kmax - 1] = math.sin(ttt)

    if ityp == 1:
        for k in range(0, kmax):
            u[0, k] = u[jm - 1, k]
            v[0, k] = v[jm - 1, k]
            u[jmax - 1, k] = u[1, k]
            v[jmax - 1, k] = v[1, k]
    else:
        for k in range(0, kmax):
            u[0, k] = u[1, k]
            v[0, k] = 0
            u[jmax - 1, k] = u[jm - 1, k]
            v[jmax - 1, k] = 0


def flow2d():
    """楕円柱まわりの低レイノルズ数流れをMAC法を用いて
       一般座標で表現されたポアソン方程式，
       およびナビエ・ストークス方程式を数値的に解く．
    """

    jmax = 62
    kmax = 31
    jm = jmax - 1
    km = kmax - 1
    x = np.zeros((jmax, kmax))
    y = np.zeros((jmax, kmax))
    grid(jmax, kmax, x, y)
    ityp = int(input("格子は周期的ですか？. はい->1, いいえ->0 : "))
    alp = int(input("迎角を入力してください. (e.g. 30): "))
    nsteps = int(input("計算打ち切りのタイムステップ数を"\
        "入力してください. (e.g. 500): "))
    it_max_poisson = int(input("ポアソン方程式を反復法で解く時の最大反復回数tを"\
        "入力してください. (e.g. 10): "))
    reynolds = int(input("レイノルズ数を入力してください. (e.g. 100): "))
    delta_t = float(input("時間間隔Δtを入力してください. (e.g. 0.01): "))
    epsilon = float(input("反復法の許容最大誤差（収束判定条件）を"\
        "入力してください. (e.g. 0.0001): "))
    acceleration_coef = float(input("流れ関数に関するポアソン方程式の近似差分"\
        "方程式をＳＯＲ法で解くための加速係数を入力してください. (e.g. 1.0): "))
    delta_t_recip = 1 / delta_t
    reynolds_recip = 1 / reynolds

    xx = np.zeros((jmax, kmax))
    xy = np.zeros((jmax, kmax))
    yx = np.zeros((jmax, kmax))
    yy = np.zeros((jmax, kmax))
    c1 = np.zeros((jmax, kmax))
    c2 = np.zeros((jmax, kmax))
    c3 = np.zeros((jmax, kmax))
    c4 = np.zeros((jmax, kmax))
    c5 = np.zeros((jmax, kmax))
    aj = np.zeros((jmax, kmax))
    u = np.zeros((jmax, kmax))
    v = np.zeros((jmax, kmax))
    p = np.zeros((jmax, kmax))
    q = np.zeros((jmax, kmax))
    d = np.zeros((jmax, kmax))

    metj(jmax, kmax, jm, km, ityp, x, y, xx, xy, yx, yy, c1, c2, c3, c4, c5, aj)
    init(jmax, kmax, u, v, p, alp)

    for n in range(0, nsteps):
        rhs(jm, km, xx, xy, yx, yy, u, v, q, delta_t_recip)
        for i in range(0, it_max_poisson):
            err = prs(jm, km, c1, c2, c3, c4, c5, p, q, acceleration_coef)
            prsbc(jmax, kmax, jm, ityp, xx, xy, c2, c3, c5, aj,
                  u, v, p, reynolds_recip)
            if err < epsilon:
                print(n, err)
                break
        ns(jm, km, xx, xy, yx, yy, c1, c2, c3, c4, c5,
           u, v, p, q, d, reynolds_recip, delta_t)
        bc(jmax, kmax, jm, ityp, u, v, alp)

    plt.contourf(x, y, p, alpha=0.5, cmap=cm.viridis)
    plt.colorbar()
    plt.contour(x, y, p, cmap=cm.viridis)
    plt.show()
    plt.quiver(x, y, u, v, color='grey')
    plt.show()


flow2d()
