% function [Bd,Dd,xO]=BD_mpm(Ad,Cd,As,Cs,K,y,u,T,delay);
%
% Estimates matrices B,D and initial state x0 using a minimum prediction error method.
%
% (Ad,Cd): is the estimated (A,C) pair of the deterministic subsystem
% (As,Cs): is the estimated (A,C) pair of the stochastic subsystem
%
%  uses the steady-state Kalman gain K computed from the previous step
%     (can do that since initial condition is also included)
%
%  u,y: input-output  data organized as row vectors,
%         [y(0) y(1)....y(T-1)], [u(0) u(1) ..........u(T-1)].
%  T: amount of data used, default: T=100.
%
%  delay : is an 1 x m  row vector, delay(i) is set to 1 force  delay from the
%        i-th input; Dd(:,i) = 0.
%    default : delay = zeros(1,m); 
%
%
%
% Written by A. Chiuso
% 1/31/2000



function [Bd,Dd,x0]=BD_mpm(Ad,Cd,As,Cs,K,y,u,T,delay);



if ((nargin > 9)| (nargin < 7))
   disp('Wrong number of input arguments');
   return 
end;

[m N]=size(u);
[p N]=size(y);

if ((nargin ==8) & T >1)
   delay = zeros(1,m);
   elseif (nargin==8)
      delay = T;
      T = 100;      
end;

if nargin == 7 
   delay = zeros(1,m);
   T = 100;
end;

% check dimensions of input matrices.

[ras cas] = size(As);
[rad cad] = size(Ad);
[rcs ccs] = size(Cs);
[rcd ccd] = size(Cd);
[rk ck] = size(K);

if cas ~=ras
   disp('As must be square')
   return
end;

if cad ~=rad
   disp('Ad must be square')
   return
end;

if cas ~=ccs
   disp('As and Cs must have the same number of columns')
   return
end;

if ras ~=rk
   disp('As and K must have the same number of rows')
   return
end;


if cad ~=ccd
   disp('Ad and Cd must have the same number of columns')
   return
end;

[ns ps]=size(K);
[pd nd]=size(Cd);
if length(delay) ~= m 
   disp('delay must have m entries')
   return
end;

if ((p ~=rcd) | (p ~=rcs))
   disp('Cd and Cs must have the same number of rows, equal to the number of outputs')
   return
end;

ns = ras; nd = rad;
n=ns+nd;
M=[];
Mh =[];
A = [Ad zeros(nd,ns) ; zeros(ns,nd) As];
C = [Cd Cs];
temp = 0;

Ac = A-[zeros(nd,ps); K]*C;
Acs = As- K*Cs;

M1 =[C; C*Ac];



ND = find(delay==0);
M2 = kron(u(ND,1:T)',eye(p));
    
MM3 = kron(u(:,1),C);

for i = 1:m,
   M3(p+1:2*p,(i-1)*n+1:i*n) = MM3((i-1)*p+1:i*p,:);
end;   
yh(:,1) = zeros(p,1);
Mh = K;

yh(:,2) = Cs*Mh*y(:,1);
for t = 2:T-1,
   M1=[C;M1*Ac];
   MM3 = MM3*Ac + kron(u(:,t),C);
   for i = 1:m,
     M3(t*p+1:(t+1)*p,(i-1)*nd+1:i*nd) = MM3((i-1)*p+1:i*p,1:nd);
     M4(t*p+1:(t+1)*p,(i-1)*p+1:i*p) = MM3((i-1)*p+1:i*p,nd+1:nd+ns)*K;
   end;   
   Mh=[Acs*Mh K];   
   clear temp;   
   temp = y(:,1:t);
   yh(:,t+1)=Cs*Mh*temp(:);  
end;

 I = find(delay); % find inputs which have to be delayed

 if ~isempty(I)
    M40=[];       
    prev=0;
    for i=1:length(I),
       t=I(i);
       M40 = [M40 M4(:,prev*p+1:(t-1)*p)];
       prev = t;
    end;
    if prev < m
     M40 = [M40 M4(:,prev*p+1:m*p)];
    end
    M4 = M40;
    clear M40;
end;    
    
M = [M1 M2-M4 M3];


yv=y(:,1:T)-yh;
yv=yv(:);

theta=M\yv;
x0=theta(1:n);


if ~isempty(I)  % there are delayed inputs
   non_del = length(ND);
   for t=1:m,
    Bd(:,t)=theta(non_del*p+n+(t-1)*nd+1:non_del*p+n+t*nd); 
   end;   
   Dd = zeros(p,m);
   for t=1:non_del,
      Dd(:,ND(t))=theta(n+(t-1)*p+1:n+t*p);
   end;
else
   for t=1:m,
    Dd(:,t)=theta(n+(t-1)*p+1:n+t*p);
    Bd(:,t)=theta(n+m*p+(t-1)*nd+1:n+m*p+t*nd);
   end;   
end;




