% function [Bd,Dd,xO]=BD_mpm_joint(A,C,K,y,u,T,delay);
%
% Estimates matrices B,D and initial state x0 using a minimum prediction error method.
%
% (A,C): is the estimated (A,C) pair of the joint deterministic-stochastic system
%
%  uses the steady-state Kalman gain K computed from the previous step
%     (can do that since initial condition is also included)
%
%  u,y: input-output  data organized as row vectors,
%         [y(0) y(1)....y(T-1)], [u(0) u(1) ..........u(T-1)].
%  T: amount of data used, default: T=100.
%
%  delay : is an 1 x m  row vector, delay(i) is set to 1 force  delay from the
%        i-th input; Dd(:,i) = 0.
%    default : delay = zeros(1,m); 
%
%
%
% Written by A. Chiuso
% 1/31/2000

function [Bd,Dd,x0]=BD_mpm_joint(A,C,K,y,u,T,delay);

if ((nargin > 7)| (nargin < 5))
   disp('Wrong number of input arguments');
   return 
end;

[m N]=size(u);
[p N]=size(y);

if ((nargin ==6) & T >1)
   delay = zeros(1,m);
   elseif (nargin==8)
      delay = T;
      T = 100;      
end;

if nargin == 5 
   delay = zeros(1,m);
   T = 100;
end;

if length(delay) ~= m 
   disp('delay must have m entries')
   return
end;

% check dimensions of input matrices.

[ra ca] = size(A);
[rc cc] = size(C);
[rk ck] = size(K);

if ca ~=ra
   disp('A must be square')
   return
end;

if ca ~=cc
   disp('A and C must have the same number of columns')
   return
end;

if ra ~=rk
   disp('A and K must have the same number of rows')
   return
end;



[ns ps]=size(K);
[pd nd]=size(C);

if (p ~=rc)
   disp('The number of rows of C must be  equal to the number of outputs')
   return
end;

n = ra;
M=[];
Mh =[];
temp = 0;

Ac = A-K*C;

M1 =[C; C*Ac];


ND = find(delay==0);
M2 = kron(u(ND,1:T)',eye(p));
MM3 = kron(u(:,1),C);

for i = 1:m,
   M3(p+1:2*p,(i-1)*n+1:i*n) = MM3((i-1)*p+1:i*p,:);
end;   
yh(:,1) = zeros(p,1);
Mh = K;

yh(:,2) = C*Mh*y(:,1);
for t = 2:T-1,
   M1=[C;M1*Ac];
   MM3 = MM3*Ac + kron(u(:,t),C);
   MM5 = MM3(:,1:n);
   MM4 = MM3*K;
   
   for i = 1:m,
     M3(t*p+1:(t+1)*p,(i-1)*n+1:i*n) = MM5((i-1)*p+1:i*p,:);
     M4(t*p+1:(t+1)*p,(i-1)*p+1:i*p) = MM4((i-1)*p+1:i*p,:);
   end;   
   Mh=[Ac*Mh K];   
   clear temp;   
   temp = y(:,1:t);
   yh(:,t+1)=C*Mh*temp(:);  
end;

I = find(delay); % find inputs which have to be delayed
 
 if ~isempty(I)
    M40=[];       
    prev=0;
    for i=1:length(I),
       t=I(i);
       M40 = [M40 M4(:,prev*p+1:(t-1)*p)];
       prev = t;
    end;
    if prev < m
     M40 = [M40 M4(:,prev*p+1:m*p)];
    end
    M4 = M40;
    clear M40;
 end;
 
M = [M1 M2-M4 M3];


yv=y(:,1:T)-yh;
yv=yv(:);

theta=M\yv;

x0=theta(1:n);
if ~isempty(I)  % there are input delayed
   ND = find(delay==0);
   non_del = length(ND);
   for t=1:m,
    Bd(:,t)=theta(non_del*p+n+(t-1)*n+1:non_del*p+n+t*n); 
   end;   
   Dd = zeros(p,m);
   for t=1:non_del,
      Dd(:,ND(t))=theta(n+(t-1)*p+1:n+t*p);
   end;
else
   for t=1:m,
    Dd(:,t)=theta(n+(t-1)*p+1:n+t*p);
    Bd(:,t)=theta(n+m*p+(t-1)*n+1:n+m*p+t*n);
   end;   
end;




