% function [Bd,Dd,xO]=BD_msm(Ad,Cd,y,u,T,delay);
%
% Estimates matrices B,D and initial state x0 using a minimum simulation error method.
%
% (Ad,Cd): is the estimated (A,C) pair of the deterministic subsystem
%
%  u,y: input-output  data organized as row vectors 
%            [y(0) y(1) ......y(T-1)], [u(0) u(1) .....u(T-1)].
%
%  T: amount of data used, default: T=100.
%
%  delay : is an 1 x m  row vector, delay(i) is set to 1 force  delay from the
%        i-th input; Dd(:,i) = 0.
%    default : delay = zeros(1,m); 
%
%
%
% Written by A. Chiuso
% 1/31/2000

function [Bd,Dd,x0]=BD_msm(Ad,Cd,y,u,T,delay);



if ((nargin > 6)| (nargin < 4))
   disp('Wrong number of input arguments');
   return 
end;

[m N]=size(u);
[p N]=size(y);


if ((nargin ==5) & T >1)
   delay = zeros(1,m);
   elseif (nargin==5)
      delay = T;
      T = 100;      
end;

if nargin == 4 
   delay = zeros(1,m);
   T = 100;
end;

if length(delay) ~= m 
   disp('delay must have m entries')
   return
end;

% check dimensions of input matrices.

[rad cad] = size(Ad);
[rcd ccd] = size(Cd);


if cad ~=rad
   disp('Ad must be square')
   return
end;


if cad ~=ccd
   disp('Ad and Cd must have the same number of columns')
   return
end;

n = cad;


if (p ~=rcd)
   disp('The number of rows of Cd must be equal to the number of outputs')
   return
end;


% construct the matrix Mt

M=[];
M1 =[Cd; Cd*Ad];

ND = find(delay==0);
M2 = kron(u(ND,1:T)',eye(p));
MM3 = kron(u(:,1),Cd);
for i = 1:m,
   M3(p+1:2*p,(i-1)*n+1:i*n) = MM3((i-1)*p+1:i*p,:);
end;   

for t = 2:T-1,
   M1=[Cd;M1*Ad];
   MM3 = MM3*Ad + kron(u(:,t),Cd);   
   for i = 1:m,
      M3(t*p+1:(t+1)*p,(i-1)*n+1:i*n) = MM3((i-1)*p+1:i*p,:);
   end;   
end;

M = [M1 M2 M3];

yv=[];
yv=y(:,1:T);
yv=yv(:);

theta=pinv(M'*M)*M'*yv;

x0=theta(1:n);

I = find(delay); % find inputs which have to be delayed

if ~isempty(I)  % there are input delayed inputs
   ND = find(delay==0);
   non_del = length(ND);
   for t=1:m,
    Bd(:,t)=theta(non_del*p+n+(t-1)*n+1:non_del*p+n+t*n); 
   end;   
   Dd = zeros(p,m);
   for t=1:non_del,
      Dd(:,ND(t))=theta(n+(t-1)*p+1:n+t*p);
   end;
else
   for t=1:m,
      Dd(:,t)=theta(n+(t-1)*p+1:n+t*p);
      Bd(:,t)=theta(n+m*p+(t-1)*n+1:n+m*p+t*n);
   end;   
end;






