% function [Ad,Bd,Cd,Dd,As,Ks,Cs,Lambda]=ort_dec(y,u,ns,nd,ks,kd,BD,T,delay,type,Aest);
%
%
% (Ad,Bd,Cd,Dd) is a state space realization of the deterministic subsystem
% (As,Ks,Cs,I) is an innovation representation of the stochastic component
%  Lambda is the variance of the innovation
%
%  y : output data as row vector y = [y(1), y(2),......,y(Ndat)]
%  u : input data as row vector u = [u(1), u(2),......,u(Ndat)]
%  ks, kd dimension of spaces in deterministic and stochastic identification
%  ns, nd are the choosen orders
%
%  BD : determination of the matrices B and D
%      BD = 'mpm' : minimum prediction error
%      BD = 'msm' : minimum simulation error
%      BD = 'msm_block' : minimum simulation error on the columns of the 
%                         estimated deterministic component
%      BD = 'proj' : projection approach (e.g.  MOESP)
%      BD = 'proj_ls' : projection approach (e.g.  MOESP)
%                       with orthonormalization
%
%  default : BD ='proj_ls'
%
%  type : identification of the deterministic part ;
%      type = 'ort' : orthogonal projection
%      type = 'obl' : oblique projection
%      type = 'cva' : Canonical Variate Analysis
%  default type = 'ort'
% 
%  delay : is an 1 x m  row vector, delay(i) is set to 1 force  delay from the
%          i-th input; Dd(:,i) = 0.
%    default : delay = zeros(1,m); 
% Aest : choice of identification method for A
%        Aest = 'TLS' total least square solution
%        Aest = 'SI' shift invariance
% default Aest ='TLS'
%
%
% Written by A. Chiuso
% 2/09/2000



function [Ad,Bd,Cd,Dd,As,Ks,Cs,Lambda]=ort_dec(y,u,ns,nd,ks,kd,BD,T,delay,type,Aest);

if ((nargin > 11) | (nargin < 2)) 
   disp('Wrong number of input arguments')
   return
end;


if ((nargin < 3) | isempty(ns))
   ns=[1:10];
end;

if ((nargin < 4) | isempty(nd))
   nd = [1:10];
end;

if ((nargin < 5) | isempty(ks) | ks < ceil(1.5*max(ns)+3) )
   ks = ceil(1.5*max(ns)+3); 
end;

if ((nargin < 6) | isempty(kd)| kd < ceil(1.5*max(nd)+3))
   kd= ceil(1.5*max(nd)+3); 

end;
if (nargin < 7  | isempty(BD))
   BD = 'proj_ls';
end;

if (nargin < 8 | isempty(T))
   T = 100;
end;

if (nargin < 9  | isempty(delay))
   delay = [0 0];
end;

if (nargin < 10  | isempty(type))
   type = 'ort';
end;

if (nargin < 11  | isempty(Aest))
   Aest = 'TLS';
end;

[m,Nu]=size(u);
[p,Ny]=size(y);

N = min(Nu,Ny)-2*kd+1;

%%% Hankel matrices:

U = Hankel(u,2*kd,N);
Y = Hankel(y,2*kd,N);

H = [U(kd*m+1:2*kd*m,:);  U(1:kd*m,:) ; Y];
[Q,R] =qr(H',0); % economy size qr decomposition 
Q=Q(:,1:2*kd*(m+p))';
R=triu(R(1:2*kd*(m+p),1:2*kd*(m+p)))';

if (strcmp(BD,'proj_ls')|strcmp(BD,'proj'))
   L= R(1:kd*m,1:kd*m);
   Phi = R(2*kd*m+p*kd+1:2*kd*(m+p),1:kd*m);
end;
if strcmp(BD,'proj')
   Phi = Phi*pinv(L);
end;


Qys = Q(2*kd*m+1:2*kd*(m+p),:);
Qyd = Q(1:2*kd*m,:);


Ys = R(2*kd*m+1:2*kd*(m+p),2*kd*m+1:2*kd*(m+p))*Qys; %Y -Yd;
% compute deterministic matrices Ad,Cd

if strcmp(type,'ort'),
   [U S V] = svd(R(kd*(2*m+p)+1:2*kd*(p+m),kd*m+1:2*kd*m)); 
   % orthogonal projection
elseif strcmp(type,'cva'),
   Yup = R(2*kd*m+kd*p+1:2*kd*(m+p),kd*m+1:2*kd*(m+p));
   Lup = chol(Yup*Yup');
   [U S V] = svd(pinv(Lup')*R(kd*(2*m+p)+1:2*kd*(p+m),kd*m+1:2*kd*m)); % cva
   U = Lup'*U;
elseif strcmp(type,'obl')
   [U S V] = svd(R(kd*(2*m+p)+1:2*kd*(p+m),kd*m+1:2*kd*m)...
      *pinv(R(kd*m+1:2*kd*m,kd*m+1:2*kd*m))*R(kd*m+1:2*kd*m,1:2*kd*m));
   % oblique projection
else
   disp('Wrong choice of method for estimation of the observability matrix')
   return
end;

if strcmp(type,'cva'),
   Ystemp = pinv(Lup')*Ys(kd*p+1:2*kd*p,:);
   syh = norm(Ystemp(p*(kd-1)+1:p*kd,:)*Ystemp(p*(kd-1)+1:p*kd,:)','fro');
else
   syh = norm(Ys(2*p*kd+1-p:2*p*kd,:)*Ys(2*p*kd+1-p:2*p*kd,:)','fro');
end;

if min(nd)<max(nd),     % select order 
      figure
      dS=diag(S)
      testo=(diag(S));
      ndef=max(max(find(testo>(max(testo)+min(testo))/2)));
      bar(((testo)));
      title(' Singular Values, deterministic subsystem')
      disp('Press any key to continue');
      pause
      for i=1:kd,
        h(i)=sum(dS(i:kd).^2);
        hh(i)=syh*kd*(kd-i+1)*((log(log(N)))^(1+0.01))/N+dS(kd)^2;  
      end;      
      ndef = sum(h>hh);   
      disp(['default=', num2str(ndef)])
      figure
      plot(h);
      hold on
      plot(hh,':');
      n=input('Select model order:(''Return'' gives default) ');
      if isempty(n)
         n=ndef;
         disp(['Order chosen to ',int2str(n)]);
      end
      close;
      close;
      else n=nd; 
end  
 
S1=S(1:n,1:n);
Gammak= U(:,1:n)*S1^(1/2);



if strcmp(Aest,'TLS') % total least squares solution 
   [U S V]=svd([Gammak(1:kd*p-p,:) Gammak(p+1:p*kd,:) ]);
   V12=V(1:n,n+1:2*n);
   V22=V(n+1:2*n,n+1:2*n); 
   Ad=-V12*pinv(V22);
elseif strcmp(Aest,'SI')
    Ad = pinv(Gammak(1:kd*p-p,:))*Gammak(p+1:p*kd,:); 
else disp('Wrong choice of estimatin method for A');    
end;  

 Cd=Gammak(1:p,:);

% filter out initial condition

z = sqrt(N)*filter_initial_gamma(Ys,[Gammak ; Gammak*Ad^kd],p);

%[Th]=n4sid(z',ns,p,ks); % compute estimate of stochastic subsystem
%if min(ns) < max(ns)
%   close
%end;

%[As,B,Cs,D,Ks,X0]=th2ss(Th);

%[rt,ct]=size(Th);
%Lambda = Th(rt-p+1:rt,1:p); % need to modify the original file N4SID to get the right Lambda
%                            % use enclosed version.
%Lambda=2

if strcmp(BD,'msm_block')
   Ncol = floor(T/(2*kd));
   Yd = R(2*kd*m+1:2*kd*(m+p),1:2*kd*m)*Qyd(:,1:Ncol);
   yd = sqrt(N)*Yd(:);
   [Bd,Dd,x0]=BD_msm_block(Ad,Cd,yd,u,2*kd,Ncol,delay);
elseif strcmp(BD,'msm')
   [Bd,Dd,x0]=BD_msm(Ad,Cd,y,u,T,delay);
elseif strcmp(BD,'mpm')
   [Bd,Dd,x0]=BD_mpm(Ad,Cd,As,Cs,Ks,y,u,T,delay); 
elseif strcmp(BD,'proj_ls')
   [Bd,Dd]=BD_proj_ls(Gammak,Phi,L,kd,delay);  
elseif strcmp(BD,'proj')
   [Bd,Dd]=BD_proj(Gammak,Phi,kd,delay); 
else disp('Wrong choice of method for estimation of B and D');
   return
end;







