import matplotlib.pyplot as plt
import numpy as np
from matplotlib import cm # for colormap
from mpl_toolkits.mplot3d import Axes3D # for plotting 3D
import peakutils # for estimating peak values

a11 = 0.4
a12 = 1.0
a13 = 15.0
a21 = 1.0
a23 = 1.0
a32 = 1.0
a31 = 0.1
b1 = 5.0
b2 = 1.0
b3 = 1.2

N1init = 1.
N2init = 2.
N3init = 3.

delta = 0.01
t = np.arange(0.,1000.+delta,delta)

##### Runge-Kutta algorithm begins here #####

def rk(t,delta,N1init,N2init,N3init):
    N1 = np.copy(t)
    N2 = np.copy(t)
    N3 = np.copy(t)
    N1[0] = N1init
    N2[0] = N2init
    N3[0] = N3init
    for i in range(0,t.shape[0]-1):
        h1 = delta*dN1dt(N1[i],N2[i],N3[i])
        k1 = delta*dN2dt(N1[i],N2[i],N3[i])
        l1 = delta*dN3dt(N1[i],N2[i],N3[i])
        h2 = delta*dN1dt(N1[i]+h1/2.,N2[i]+k1/2.,N3[i]+l1/2.)
        k2 = delta*dN2dt(N1[i]+h1/2.,N2[i]+k1/2.,N3[i]+l1/2.)
        l2 = delta*dN3dt(N1[i]+h1/2.,N2[i]+k1/2.,N3[i]+l1/2.)
        h3 = delta*dN1dt(N1[i]+h2/2.,N2[i]+k2/2.,N3[i]+l2/2.)
        k3 = delta*dN2dt(N1[i]+h2/2.,N2[i]+k2/2.,N3[i]+l2/2.)
        l3 = delta*dN3dt(N1[i]+h2/2.,N2[i]+k2/2.,N3[i]+l2/2.)
        h4 = delta*dN1dt(N1[i]+h3,N2[i]+k3,N3[i]+l3)
        k4 = delta*dN2dt(N1[i]+h3,N2[i]+k3,N3[i]+l3)
        l4 = delta*dN3dt(N1[i]+h3,N2[i]+k3,N3[i]+l3)
        N1[i+1] = N1[i]+(h1+2.*h2+2.*h3+h4)/6.
        N2[i+1] = N2[i]+(k1+2.*k2+2.*k3+k4)/6.
        N3[i+1] = N3[i]+(l1+2.*l2+2.*l3+l4)/6.
    return N1,N2,N3

def dN1dt(N1,N2,N3):
    return (b1-a11*N1-a12*N2-a13*N3)*N1

def dN2dt(N1,N2,N3):
    return (-b2+a21*N1-a23*N3)*N2

def dN3dt(N1,N2,N3):
    return (-b3+a31*N1+a32*N2)*N3

##### Runge-Kutta algorithm ends here #####

def timeseries():
	#For displaying the time-series
	plt.figure(1,figsize=(12,6))
	val = rk(t,delta,N1init,N2init,N3init)
	for i in range(0,3):
	    plt.plot(t,val[i],label='N%1i'%(i+1))
	plt.xlim([0.,100.])
	plt.xlabel('Time (s)')
	plt.ylabel('Population Density')
	plt.legend(loc=2)
	plt.show()

def timeseries_sensitive_to_initial():
	#For demonstrating the sensitivity of the system to initial conditions
	val = rk(t,delta,N1init,N2init,N3init)

	plt.figure(1,figsize=(12,6))
	for i in range(0,3):
	    plt.plot(t,val[i],label='N%1i'%(i+1))

	#The next code are additional cases where N1init is added with 0.01
	val = rk(t,delta,N1init+0.001,N2init,N3init)
	for i in range(0,3):
	    plt.plot(t,val[i],label='N%1i_b'%(i+1),linewidth=1.0)

	plt.xlim([900.,1000.])
	plt.legend(loc=1)
	plt.show()

def strangeattractor():
	#For demonstrating the strange attractors in 3D.
	val = rk(t,delta,N1init,N2init,N3init)

	fig = plt.figure(figsize=(8,8))
	ax = fig.gca(projection='3d')
	surf = ax.scatter(val[0],val[1],val[2],c=t,s=1.)
	fig.colorbar(surf, shrink=0.5, aspect=5)
	ax.set_xlabel('N1')
	ax.set_ylabel('N2')
	ax.set_zlabel('N3')
	ax.set_xlim([0.,12.])
	ax.set_ylim([0.,12.])
	ax.set_zlim([0.,6.])
	plt.show()

def bifurcation():
	#For demonstrating bifurcation.
	plt.clf() #Necessary to clear the figure at the first instance.
	fig = plt.figure(1,figsize=(10,8))
	delta = 0.05
	t = np.arange(0.,5000.+delta,delta)
	global a13
	for a13 in np.arange(5.,7.,0.1):
		val = rk(t,delta,N1init,N2init,N3init)[:]
		ind = peakutils.indexes(val[2][10000:100000],thres = 0.,min_dist=1)
		x = np.empty(ind.shape[0],dtype=np.float32)
		x[:] = a13
		plt.plot(x,val[2][10000:][ind],'k.',markersize=0.5, alpha=0.5)
	plt.xlim([5.,8.])
	plt.xlabel(r"$a_{13}$")
	plt.ylabel(r"Maximas of $N_3$")
	plt.show()

def main():
	#Remove the '#' in the first character to run the modules.

	timeseries()
#	timeseries_sensitive_to_initial()
#	strangeattractor()
#	bifurcation() #Warning: This will take a long time to compute depending on the system.

if __name__=="__main__":
	main()
